/* -*-C-*-
 ##############################################################################
 #
 # File:        demo/rpmtrig.c
 # RCS:         $Id: rpmtrig.c,v 1.18 1998/07/20 22:59:29 ericb Exp $
 # Description: Host program for doing RPM arm/trigger
 # Created:     October 20, 1995
 # Language:    C
 # Package:     E1432
 #
 # Copyright (C) 1995 - 1998, Hewlett-Packard Company, all rights reserved.
 #
 ##############################################################################
 #
 # This is a host program that will demonstrate RPM arming/triggering on a
 # a single E1432.  It will display time waveforms for all channels.
 # It will print out the RPM values of both tach channels at the trigger 
 # points.
 #
 # The equipment required for this demo is:
 # 	1. VXI card cage with MXI interface to host
 #  	2. one E1432 with at least 4 input channels and a tachometer option AYD
 #
 # Set the logical address of the E1432 to the address specified in the LA1 
 # defined below (default 8).
 #
 # Hook up a sweeping signal generator as such:
 #    output set to 2 volts peak sine wave.
 #    tee output to go to the channel 1 input and both tachometer inputs 
 #
 # To run the measurement:
 #    start the program
 #    sweep the frequency between 5Hz(300 RPM) and 105Hz (6300 RPM)
 #    over 30 seconds
 #
 # The program should display the waveform in the channel 1 display on
 # the screen.  The measurement pre-arms at 500 RPM and begins triggering and
 # updating the display at 700 RPM and every 100 RPM interval after that 
 # until 6000 rpm.  If the trigger delay and trigger mode are left at the
 # defaults ( -BLOCK_SIZE/2 and E1432_TACH_EDGE_TRIGGER) the center point of
 # the block of data will remain stationary since we are triggering on a
 # tach edge and taking data +/- BLOCK_SIZE/2 on either side of that trigger
 # point.  In addition, the arming RPM points, RPM at trigger point of arming
 # tach channel, and average RPM of the block on other tach channel are 
 # displayed on the terminal window.
 #
 # Note that the RPMs for the two channels are not exactly the same.  The
 # RPM of the arm/trigger channel is calculated using a quadratic fitting 
 # method across tach edge times near the trigger point, while the second
 # channel's RPM is calculated by taking the average RPM for the block.
 #
 # Altering some of the default measurement parameters below will change the
 # nature of the display.  In particular, changing the trigger mode to
 # E1432_AUTO_TRIGGER will produce a display whose center is the exact point
 # at which the RPM reached the arming RPM and won't be lined up with a zero
 # crossing of the tach.  
 #
 # Revisions:
 #
 #
 ##############################################################################
 */
#include <stdio.h>
#include <stdlib.h>
#include <sys/termio.h>
#include <unistd.h>
#include "e1432.h"
#include "xplot.h"

/*#define EXT_TRIG*/
/*#define TRAILER */

/* defines for size of XPLOT windows */
#define WIDTH           140
#define HEIGHT          80

#define WARNING_MAX	100

/* default module addresses */
#define LA1		8

/* default measurement parameters */
#define BLOCK_SIZE	1024
#define SPAN		20000.0
#define RANGE		2.0
#define RPM_PRE_ARM	500.0
#define RPM_LOW		700.0
#define RPM_HIGH	6000.0
#define RPM_INTERVAL	100.0

#if 0
/* trigger at the next tach edge after arm rpm reached */
#define TRIGGER_MODE	E1432_TACH_EDGE_TRIGGER
#else
/* trigger at exact arm rpm */
#define TRIGGER_MODE	E1432_AUTO_TRIGGER
#endif

#if 1
/* trigger in the middle of the block */
#define TRIGGER_DELAY	-BLOCK_SIZE/2
#else
/* trigger at the beginning of the block */
#define TRIGGER_DELAY	0
#endif

/* Wrap this around all the many function calls which might fail */
#define	DEBUG(s)	s
#ifdef	__lint
#define	CHECK(func)	\
do {\
    int _s = (func);\
    if (_s < 0)\
    {\
	DEBUG((void) printf("Error: %s returned %d\n", #func, _s));\
	return _s;\
    }\
} while (func)
#else
#define	CHECK(func)	\
do {\
    int _s = (func);\
    if (_s < 0)\
    {\
	DEBUG((void) printf("Error: %s returned %d\n", #func, _s));\
	return _s;\
    }\
} while (0)
#endif

/*****************************************************************************
 *
 * Check for key being pressed.  Return  if key pressed, 0 if no key.
 *
 ****************************************************************************/
static char key_pressed(void) 
{
  char ch;
  LONGSIZ32 numChars;

  (void) ioctl(0, FIONREAD, &numChars);

  if(numChars) {
      (void) read(0, &ch, 1);
      return(ch);
  }
  return 0;
}


int
main(void)
{
    int  i, j, nchan;
    LONGSIZ32 count;
    LONGSIZ32 num = 0;
    SHORTSIZ16 la = LA1;
    SHORTSIZ16 status;
    SHORTSIZ16 chan_list[16];
    SHORTSIZ16 tach_list[E1432_TACH_CHANS];
    SHORTSIZ16 inputs, tachs, tach0, tach1, arm, nonarm;
    E1432ID hw;
    float rpm;
    struct e1432_hwconfig cf;
    SHORTSIZ16 error;
    FLOATSIZ32 *data[16];
    long points = BLOCK_SIZE;
    char *plotid[16];
    int row, col;  
    char geometry[80];
    char title[80];
    float temp, floatRange = RANGE;
    char *semabinloc = "/opt/e1432/lib/sema.bin";
    SHORTSIZ16 meas_state;
    struct e1432_trailer trailer;
    SHORTSIZ16 warning[WARNING_MAX];
    unsigned long warningCount;

    if(e1432_init_io_driver()) {
       (void) printf("e1432_init_io_driver() failed\n");
       exit(0);
    }

    CHECK(e1432_print_errors(0));

    /* install the downloadable code */
    (void) printf("Checking for E1432 with firmware at logical address %d ... ",
		  la);
    (void)fflush(stdout);
    error = e1432_get_hwconfig(1, &la, &cf);
    CHECK(e1432_print_errors(1));
    if (error)
    {
	(void) printf("Not found.\n\n");
	(void) printf("Installing firmware from %s into E1432 at la %d ... ",
		      semabinloc, la);
	(void) fflush(stdout);
	error = e1432_install(1, &la, 0, semabinloc);
	if (error)
	{
	    (void) printf("\ne1432_install failed and returned error %s\n");
	    exit(0);
	}
	(void) printf("Done.\n");
    }
    else
    {
	(void) printf("Found.\n");
    }
    (void) printf("\n");

    /* assign channel numbers */
    CHECK(e1432_assign_channel_numbers(1, &la, &hw));

    /* Create channel group for all input channels */
    error = e1432_get_hwconfig(1, &la, &cf);
    if(error) {
        (void) printf("error in e1432_get_hwconfig(): %d\n", error);
        exit(1);
    }
    nchan = cf.input_chans;

    (void) printf("Found %d input channels\n", nchan);
    (void) printf("Wait for initial rpm readout before sweeping input\n");

    for(i=0; i < nchan; i++) {
       chan_list[i] = E1432_INPUT_CHAN(i+1);
    }

    inputs = e1432_create_channel_group(hw, nchan, chan_list);
    if (inputs >= 0)
    {
	(void) printf("e1432_create_channel_group for inputs returned %d\n",
		      inputs);
	exit(1);
    }

    /* Initialize hardware things */
    CHECK(e1432_set_analog_input(hw, inputs,
				     E1432_INPUT_MODE_VOLT,
				     E1432_INPUT_SOURCE_BNC,
				     E1432_ANTI_ALIAS_ANALOG_ON,
				     E1432_COUPLING_DC, floatRange));

    CHECK(e1432_set_blocksize(hw, inputs, BLOCK_SIZE));
    CHECK(e1432_set_data_size(hw, inputs, E1432_DATA_SIZE_32));

    /* initialize buffers and XPLOT stuff */
    row = col = 0;
    temp = (float)(points - 1);

    for(i=0; i < nchan; i++) {
        data[i] = (FLOATSIZ32 *)malloc(sizeof(FLOATSIZ32) * points);
        if(!data[i]) {
          (void) printf("Couldn't malloc data array of %d points\n",
			points);
          exit(1);
        }

	/* initialize so XPLOT doesn't throwup */
	for(j=0; j < points; j++) {
	   data[i][j] = 0.0;
	}

        (void) sprintf(geometry, "%dx%d+%d+%d", WIDTH, HEIGHT,
		       (WIDTH + 20) * col, (HEIGHT + 40) * row);
	(void) sprintf(title, "Channel %d", i + 1);

        plotid[i] = xplot_init_plot(data[i], points, temp, floatRange, 
			-floatRange, GENERIC_TRACE, geometry, title);
        col++;
        if((col % 8) == 0) {
            col = 0;
            row++;
        }

        xplot_change_yautoscale(plotid[i], 0);
        xplot_set_xscale(plotid[i], (float)0.0, temp);
        xplot_set_yscale(plotid[i], floatRange, -floatRange);
        xplot_change_xlabel(plotid[i], "Samples");
        xplot_change_ylabel(plotid[i], "Volts");
        xplot_repaint(plotid[i]); 
    }

    /* assign tach channels */
    tach0 = tach_list[0] = E1432_TACH_CHAN(1);
    tach1 = tach_list[1] = E1432_TACH_CHAN(2);
    tachs = e1432_create_channel_group(hw, 2, tach_list);
    if (tachs >= 0)
    {
	(void) printf("create_channel_group for tachs returned %d\n",
		      tachs);
	exit(1);
    }

/* default is to have tach0 as the arming channel, tach1 as the non-arm */
#if 1
    arm = tach0;
    nonarm = tach1;
#else
    arm = tach1;
    nonarm = tach0;
#endif

    CHECK(e1432_set_span(hw, inputs, SPAN));
    CHECK(e1432_set_data_mode(hw, inputs, E1432_DATA_MODE_OVERLAP_BLOCK));

    /* set up inputs arm trigger stuff */
#ifdef EXT_TRIG
    CHECK(e1432_set_trigger_channel(hw, arm, E1432_CHANNEL_OFF));
    CHECK(e1432_set_trigger_channel(hw, nonarm, E1432_CHANNEL_ON));
#else
    CHECK(e1432_set_trigger_channel(hw, arm, E1432_CHANNEL_ON));
    CHECK(e1432_set_trigger_channel(hw, nonarm, E1432_CHANNEL_OFF));
#endif

    CHECK(e1432_set_trigger_level(hw, tachs, E1432_TRIGGER_LEVEL_LOWER, 0.0));
    CHECK(e1432_set_trigger_level(hw, tachs, E1432_TRIGGER_LEVEL_UPPER, 0.0));
    CHECK(e1432_set_arm_mode(hw, inputs, E1432_ARM_RPM_RUNUP)); 
    CHECK(e1432_set_pre_arm_mode(hw, arm, E1432_ARM_RPM_RUNUP)); 
    CHECK(e1432_set_arm_channel(hw, arm, E1432_CHANNEL_ON)); 
    CHECK(e1432_set_pre_arm_rpm(hw, arm, RPM_PRE_ARM)); 
    CHECK(e1432_set_rpm_low(hw, arm, RPM_LOW)); 
    CHECK(e1432_set_rpm_high(hw, arm, RPM_HIGH)); 
    CHECK(e1432_set_rpm_interval(hw, arm, RPM_INTERVAL)); 

#if 0
    CHECK(e1432_set_internal_debug(hw, arm, 0x400));
#endif 

    /* set block start relative to trigger point */
    CHECK(e1432_set_trigger_delay(hw, inputs, TRIGGER_DELAY));

    /* trigger mode */
    CHECK(e1432_set_auto_trigger(hw, inputs, TRIGGER_MODE)); 

#ifdef TRAILER
    CHECK(e1432_set_append_status(hw, inputs, E1432_APPEND_STATUS_ON)); 
#endif

    /* stop any previous measurement */
    CHECK(e1432_finish_measure(hw, inputs)); 

    (void) printf("Continuous measurement, press Cntl C to terminate\n\n");

    /* Start measurement */
    CHECK(e1432_init_measure(hw, inputs));

    (void)sleep(1);

    CHECK(e1432_get_current_rpm(hw, arm, &rpm));
    (void) printf("arm channel rpm = %-6.6g\n\n", rpm);

    (void) printf("Waiting for pre-arm RPM of %g ... ", RPM_PRE_ARM);
    (void) fflush(stdout);

    for(meas_state = 0; meas_state < E1432_MEAS_STATE_IDLE; )
    {
        CHECK(e1432_get_meas_state(hw, arm, &meas_state));
	status = e1432_block_available(hw, inputs);
	if(status) break;	/* go on if data is available */
    }
    (void) printf("PRE-ARMED.\n\n");

    (void) printf("Waiting for initial arming RPM of %g ... ",
		  RPM_LOW);
    (void) fflush(stdout);

    for(meas_state = 0; meas_state < E1432_MEAS_STATE_TRIGGER; )
    {
        CHECK(e1432_get_meas_state(hw, arm, &meas_state));
	status = e1432_block_available(hw, inputs);
	if(status) break;	/* go on if data is available */
    }
    (void) printf("ARMED.\n\n");

    (void) printf("TRIG RPM	TACH2 RPM\n");

    /* main measurement loop */
    while(1) 
    {
        /* Wait for block available , handling triggers */
	do
	{
            CHECK(e1432_read_register(hw, chan_list[0], 	
					E1432_IRQ_STATUS2_REG, &status));
	
            if (status < 0) 
	    {
	        (void) printf("status error %d \n", status);
	        exit(1);
            }
	    
	    if(status & E1432_IRQ_MEAS_WARNING)
	    {
		/* read out all measurement warnings */
		while(status & E1432_IRQ_MEAS_WARNING)
		{
		    CHECK(e1432_get_meas_warning(hw, inputs, warning, 
						WARNING_MAX, &warningCount));
	   
		    if(warningCount)
		    {
	       		(void) printf("%d Warning", warningCount);
			if(warningCount > 1) (void) printf("s");
			(void) printf(":\n");
		    }

           	    for(i=0; i < warningCount; i++)
	   	    {
			(void) printf("    %s\n",
				      e1432_get_warning_string(warning[i]));
	   	    }

		    CHECK(e1432_read_register(hw, chan_list[0], 	
					E1432_IRQ_STATUS2_REG, &status));
		}
	    }

            CHECK(e1432_get_meas_state(hw, arm, &meas_state));
	    if(meas_state == E1432_MEAS_STATE_TESTED)
	    {
		(void) printf("Measurement finished.\n");
		exit(0);
	    }
	    
	    status = e1432_block_available(hw, inputs);
            if (status < 0) 
	    {
	        (void) printf("e1432_block_available() error %d \n", status);
	        exit(1);
            }

	}while (status == 0); 

	CHECK(e1432_get_data_rpm(hw, arm, &rpm));
	(void) printf("%-6.6g		", rpm);
	CHECK(e1432_get_data_rpm(hw, nonarm, &rpm));
	(void) printf("%-6.6g\n", rpm);
        
	for(i=0; i < nchan; i++) 
	{	/* read time data */
            error = e1432_read_float32_data(hw, chan_list[i],
		    E1432_TIME_DATA, data[i], points, &trailer, &count);
            if(error) 
	    {
   	        (void) printf("read time data chan %d failed with error %d\n",
					i + 1, error);
	        exit(1);
	    }

#ifdef TRAILER
	    if(i==0)
	        (void) printf("trailer rpm:  %g   %g\n",
			      trailer.rpm1, trailer.rpm2);
#endif

            xplot_check_events(plotid[i]);
            xplot_data_update(plotid[i]); 
        }

	if((++num % 20) == 0)
    	    (void) printf("\nTRIG RPM	TACH2 RPM\n");

	(void) fflush(stdout);

	if(key_pressed())
	{
	    (void) printf("Measurement stopped\n");
    	    CHECK(e1432_finish_measure(hw, inputs)); 
	    exit(0);
	}
    }

}
